<?php

namespace App\Models;

use App\Models\Scopes\CategoryType;
use App\Trait\Imageable;
use Illuminate\Database\Eloquent\Attributes\ScopedBy;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

#[ScopedBy(CategoryType::class)]
class Category extends Model
{
    /** @use HasFactory<\Database\Factories\CategoryFactory> */
    use HasFactory, Imageable;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'status',
        'parent_id',
        'agent_id',
        'order',
        'type',
    ];

    protected $casts = [
        'status' => 'boolean',
    ];

    protected $with = ['image'];

    public function parent(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(Category::class, 'parent_id');
    }

    public function packages(): HasMany
    {
        return $this->hasMany(Package::class);
    }

    public function blogs(): HasMany
    {
        return $this->hasMany(Blog::class);
    }

    public function faqs(): HasMany
    {
        return $this->hasMany(Package::class);
    }

    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function scopeParent($query)
    {
        return $query->whereNull('parent_id');
    }

    public function scopeChild($query)
    {
        return $query->whereNotNull('parent_id');
    }

    public function scopeGetType($query, $type)
    {
        return $query->active()->whereType($type)->withoutGlobalScope(CategoryType::class);
    }
}
